//	TorusGames-Common.h
//
//	Platform-independent definitions for Torus Games's internal code.
//	The internal code doesn't know or care what platform (iOS or macOS) it's running on.
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#pragma once

#include "GeometryGames-Common.h"
#include "GeometryGamesSpinGroup.h"
#include "GeometryGamesUtilities-SIMD.h"
#include "GeometryGamesFauxSimd.h"	//	Metal file would need explicit path


//	Use a mouse-based interface on macOS, but
//	use a touch-based interface on iOS.

#if TARGET_OS_IOS
#define TORUS_GAMES_2D_TOUCH_INTERFACE
#endif

#if TARGET_OS_OSX
#define TORUS_GAMES_2D_MOUSE_INTERFACE
//#define TORUS_GAMES_2D_TOUCH_INTERFACE
#endif

#if ! defined(TORUS_GAMES_2D_TOUCH_INTERFACE) && ! defined(TORUS_GAMES_2D_MOUSE_INTERFACE)
#error Must define either touch interface or  mouse interface.
#endif
#if   defined(TORUS_GAMES_2D_TOUCH_INTERFACE) &&   defined(TORUS_GAMES_2D_MOUSE_INTERFACE)
#error Cannot define both touch interface and mouse interface.
#endif


//	For the Shape of Space talk, suppress the Help files
//	when the user chooses the Intro2D game, and
//	offset Tic-Tac-Toe by half a square.
//#define TORUS_GAMES_FOR_TALK
#ifdef TORUS_GAMES_FOR_TALK
#warning TORUS_GAMES_FOR_TALK is enabled

//#define USE_CUSTOM_KLEIN_BOTTLE_CROSSWORD_FOR_TALK
#ifdef USE_CUSTOM_KLEIN_BOTTLE_CROSSWORD_FOR_TALK
#warning USE_CUSTOM_KLEIN_BOTTLE_CROSSWORD_FOR_TALK is enabled
#endif

//#define USE_CUSTOM_TORUS_CROSSWORD_FOR_TALK
#ifdef USE_CUSTOM_TORUS_CROSSWORD_FOR_TALK
#warning USE_CUSTOM_TORUS_CROSSWORD_FOR_TALK is enabled
#endif

#endif	//	TORUS_GAMES_FOR_TALK


//	CREATE_CROSSWORD_FOR_TALK provides a completely empty board,
//	for use when designing puzzles by hand.
//#define CREATE_CROSSWORD_FOR_TALK
#ifdef CREATE_CROSSWORD_FOR_TALK
#warning CREATE_CROSSWORD_FOR_TALK is enabled
#endif


//	Use MAKE_MANUAL_SCREENSHOT to generate individual screenshots,
//	for example for the Torus Games web page on geometrygames.org.
//#define MAKE_MANUAL_SCREENSHOT
#ifdef MAKE_MANUAL_SCREENSHOT
#warning MAKE_MANUAL_SCREENSHOT is enabled
#define GAME_CONTENT_FOR_SCREENSHOT	//	used for both MAKE_MANUAL_SCREENSHOT and MAKE_SCREENSHOTS
#endif

//	Use PREPARE_FOR_SCREENSHOT while looking
//	for an aesthetically pleasing view of the 3D Maze, and then...
//#define PREPARE_FOR_SCREENSHOT
#ifdef PREPARE_FOR_SCREENSHOT
#warning PREPARE_FOR_SCREENSHOT is enabled
#endif
//	...use MAKE_SCREENSHOTS to generate all the screenshots
//	in all the languages for all the games
//	(including that aesthetically pleasing view of the 3D Maze).
//#define MAKE_SCREENSHOTS
#ifdef MAKE_SCREENSHOTS
#warning MAKE_SCREENSHOTS is enabled
#define GAME_CONTENT_FOR_SCREENSHOT	//	used for both MAKE_MANUAL_SCREENSHOT and MAKE_SCREENSHOTS
#endif

#if (defined(MAKE_SCREENSHOTS) && defined(MAKE_MANUAL_SCREENSHOT))
#error MAKE_SCREENSHOTS and MAKE_MANUAL_SCREENSHOT shouldn't be enabled at the same time.
#endif

//	Don't roughen textures when making the app icon.
//#define MAKE_APP_ICON
#ifdef MAKE_APP_ICON
#warning MAKE_APP_ICON is enabled
#endif

//	(Semi-)automate creation of game choice menu icons
//#define MAKE_GAME_CHOICE_ICONS
#ifdef MAKE_GAME_CHOICE_ICONS
#warning MAKE_GAME_CHOICE_ICONS is enabled
#endif

//	For use in the Flandrau Science Center at the University of Arizona,
//		- include an inactivity timeout
//		- suppress the Practice Board
//		- suppress Repeating view mode
//		- suppress the Sound Effects switch
//#define SCIENCE_CENTER_VERSION
#ifdef SCIENCE_CENTER_VERSION
#warning SCIENCE_CENTER_VERSION is enabled
#error Uncomment the warning in the else clause below, if changing the Bundle ID
#else
//#warning If I changed the Bundle ID to org.geometrygames.TorusGames-museum...
//	...then change it back to org.geometrygames.TorusGames-iOS
//	(and then when recommenting this warning, uncomment the error message above)
#endif

#ifdef SCIENCE_CENTER_VERSION
#warning Be sure to enable DISABLE_TEXT_SELECTION_AT_FLANDRAU in GeometryGamesWebViewController.m \
	and then disable it again after compiling the Flandrau Torus Games.
#endif


//	Figures for the 3rd edition of The Shape of Space

//		Shape of Space chessboards
//
//		1	Figure 2.6
//		2	Figure 2.7
//		3	Figure 2.8
//		4	Figure 4.8
//
//#define SHAPE_OF_SPACE_CHESSBOARD	1
#ifdef SHAPE_OF_SPACE_CHESSBOARD
#warning SHAPE_OF_SPACE_CHESSBOARD is enabled
#endif

//		Shape of Space tic-tac-toe boards
//
//		1	board with grid lines
//		2	X's and O's over transparent background
//
//		It's easier to make the various tic-tac-toe boards
//		manually at run time than to define them in code ahead of time.
//
//#define SHAPE_OF_SPACE_TICTACTOE	2
#ifdef SHAPE_OF_SPACE_TICTACTOE
#warning SHAPE_OF_SPACE_TICTACTOE is enabled
#endif



//	Don't let the user open more than one window at once.  
//	On my 2006 iMac's Radeon X1600, opening 2 or 3 windows 
//	with depth buffers, or 3 or 4 windows without depth buffers, 
//	suddenly brings the computer to a near standstill.
//	On my 2008 MacBook's GeForce 9400M the problem seems not to occur.
//	To enable multiple-window support for testing purposes, 
//	uncomment the following line.
//#define ALLOW_MULTIPLE_WINDOWS
#ifdef ALLOW_MULTIPLE_WINDOWS
#warning ALLOW_MULTIPLE_WINDOWS is enabled
#endif


//	Square roots
#define ROOT2		1.41421356237309504880	//	√2
#define ROOT3		1.73205080756887729353	//	√3
#define ROOT_HALF	0.70710678118654752440	//	√½


//	Goal window width as fraction of total goal width
#define GOAL_WINDOW_FRACTION	0.75

//	Obviously a cube has six faces, but let's #define a constant
//	for that number anyhow, so we'll know what parts of the code depend on it.
#define NUM_FRAME_WALLS	6

//	A 3D rotation or scroll coasts to a gradual stop
//	after the user has lifted his/her finger.

#define TRANSLATIONAL_COASTING_DECELERATION	2.0			//	distance / sec²
#define MAX_TRANSLATIONAL_COASTING_DISTANCE	4.0			//	distance
#define MAX_TRANSLATIONAL_COASTING_SPEED	( sqrt(2.0 * TRANSLATIONAL_COASTING_DECELERATION * MAX_TRANSLATIONAL_COASTING_DISTANCE) )	//	distance / sec

#define ROTATIONAL_COASTING_DECELERATION	2.0			//	radians / sec²  for half-angle
#define MAX_ROTATIONAL_COASTING_ANGLE		(0.5 * PI)	//	radians         for half-angle
#define MAX_ROTATIONAL_COASTING_SPEED		( sqrt(2.0 * ROTATIONAL_COASTING_DECELERATION * MAX_ROTATIONAL_COASTING_ANGLE) )	//	radians / sec  for half-angle


//	Number of texture repetitions for each glide symmetry axis in Klein bottle
#define NUM_KLEIN_AXIS_TEXTURE_REPETITIONS	8


//	Buffer size for preparing status message.
//	If one of the game attempted to set a longer message
//	(for example, if a Word Search puzzle used more words
//	than the buffer could hold) nothing terrible would happen.
//	The message would get truncated at the penultimate position
//	and a terminating zero would get written into the last position.
//
#define STATUS_MESSAGE_BUFFER_SIZE	1024	//	number of Char16's, including terminating zero


//	When composing kana, e.g. 's' + 'u' = 'す',
//	let 0 represent no pending character.
#define NO_PENDING_CHARACTER	0

//	π
#define PI		3.14159265358979323846
#define TWOPI	(2.0 * PI)

//	Some games refer to recent cursor positions
//	to determine the cursor heading.
#define DRAG_HISTORY_DEPTH	64


//	Handy utilities.
//	(Caution:  Arguments may be evaluated twice!)
#ifndef MIN
	#define MIN(A,B)	((A) < (B) ? (A) : (B))
#endif
#ifndef MAX
	#define MAX(A,B)	((A) > (B) ? (A) : (B))
#endif
#ifndef ABS
	#define ABS(A)		((A) >= 0 ? (A) : (-(A)))
#endif
#define SIGN(A)			((A) != 0 ? ((A) > 0 ? (+1) : (-1)) : 0)


//	Enumerate available games.
typedef enum
{
	GameNone,

	//	2D games
	Game2DIntro,
	Game2DTicTacToe,
	Game2DGomoku,
	Game2DMaze,
	Game2DCrossword,
	Game2DWordSearch,
	Game2DJigsaw,
	Game2DChess,
	Game2DPool,
	Game2DApples,
	
	//	3D games
	Game3DTicTacToe,
	Game3DMaze,

	NumGameTypes

} GameType;

//	Enumerate available topologies.
typedef enum
{
	//	2D topologies
	Topology2DTorus,		//	torus
	Topology2DKlein,		//	Klein bottle
	
	//	3D topologies
	Topology3DTorus,		//	3-torus
	Topology3DKlein,		//	Klein space with x ↔ -x flip in the y direction
	Topology3DQuarterTurn,	//	quarter turn space with 1/4 CCL turn in z direction
	Topology3DHalfTurn,		//	half turn space with half turn in z direction
	
	NumTopologyTypes
	
} TopologyType;

//	Enumerate available views.
typedef enum
{
	//	View a single copy of the torus.
	//
	//		Note:
	//		I had originally called this the fundamental domain mode,
	//		but a fundamental domain has a well-defined boundary,
	//		whereas when people play the games, and especially
	//		when they scroll the board, they get the impression
	//		of a game board with no boundary at all.
	//		So I renamed this the basic mode.
	//
	//	For 3D games, its3DDesiredAperture and its3DCurrentAperture must both be zero.
	//
	ViewBasicLarge,
	
	//	View a single copy of the torus, but at 1/3 size,
	//	as a transition to the repeating mode.
	//
	//	Used in 2D games only.
	//
	ViewBasicSmall,
	
	//	View the universal cover.  Well, really just a 3×3 portion of it.
	//
	//	For 3D games, assumes the frame cell is axis aligned.
	//	Any of the 6·4 = 24 possible alignments is fine.
	//	its3DDesiredAperture must be 1.
	//	its3DCurrentAperture is typically also 1,
	//	but may be less than 1 during transitions.
	//
	ViewRepeating
} ViewType;

//	Enumerate hand cursor modes, for 2D games.
typedef enum
{
	HandNone,	//	We don't have the cursor.
	HandFree,	//	The cursor is moving freely within our space.
	HandScroll,	//	The cursor is scrolling the space relative to the window.
	HandDrag	//	The cursor is dragging an object within the space.
} HandStatus2D;

//	To place an object (either a sprite or a cursor)
//	in the fundamental domain of a 2D games, we apply
//	a dilation, a rotation, a possible side-to-side reflection
//	and a translation, in that order.
typedef struct
{
	double	itsH,		//	horizontal translation, in range [-0.5, +0.5], left to right
			itsV;		//	 vertical  translation, in range [-0.5, +0.5], bottom to top
	bool	itsFlip;	//	reflection, always side-to-side
	double	itsAngle,	//	rotation, in radians counterclockwise
			itsSizeH,	//	horizontal dilation
			itsSizeV;	//	vertical dilation
} Placement2D;

//	The fundamental domain itself may be translated and flipped
//	within the window, but not rotated or dilated.
typedef struct
{
	double	itsH,		//	horizontal offset, in range [-0.5, +0.5]
			itsV;		//	 vertical  offset, in range [-0.5, +0.5]
	bool	itsFlip;	//	reflection, always side-to-side
} Offset2D;

//	When a scroll or rotation ends,
//	let the game board coast gradually to a stop.
typedef enum
{
	CoastingNone,
	Coasting2DTranslation,
	Coasting3DTranslation,
	Coasting3DRotation
} CoastingType;


//	What's happening during idle time?
typedef enum
{
	SimulationNone,	//	typically the case, if nothing special is happening

	Simulation2DTicTacToeWaitToMove,
	Simulation2DTicTacToeWaitToDisplayWin,
	Simulation2DTicTacToeFlash,
	Simulation2DGomokuChooseComputerMove,
	Simulation2DGomokuWaitToDisplayWin,
	Simulation2DGomokuFlash,
	Simulation2DMazeFlash,
	Simulation2DCrosswordFlash,
	Simulation2DWordSearchFlash,
	Simulation2DChessWaitToMove,
	Simulation2DChessChooseComputerMove,
	Simulation2DChessAnimateComputerMove,
	Simulation2DPoolRollBalls,
	Simulation2DPoolPlaceCueStick,
	Simulation2DPoolPauseBeforeShooting,

	Simulation3DTranslationSnapToGrid,
	Simulation3DRotationSnapToAxes,
	Simulation3DExitTilingMode,
	Simulation3DResetGameAsDomainPart1,
	Simulation3DResetGameAsDomainPart2,
	Simulation3DResetGameAsDomainPart3,
	Simulation3DResetGameAsTilingPart1,
	Simulation3DResetGameAsTilingPart2,
	Simulation3DResetGameAsTilingPart3,
	Simulation3DTicTacToeSelectNodePart1,
	Simulation3DTicTacToeSelectNodePart2,
	Simulation3DTicTacToeSelectNodePart3,
	Simulation3DTicTacToeWaitToMove

} SimulationType;


//	In a 3D game, what is the current drag for?
typedef enum
{
	DragNone,
	DragRotate,		//	rotate the frame cell in the world
	DragScroll,		//	scroll the tiling in the framecell
	DragContent,	//	drag the game's content
	DragIgnore		//	ignore the drag because a simulation is in progress
} DragType3D;


//	In a 3D game, when the user taps the display
//	with his/her finger (iOS) or clicks with the mouse (macOS or Windows),
//	we'll need to decide which object (if any) gets hit.
//	Interpret the tap or click as a ray that begins at
//	the user's eye, (0, 0, -1, 1) in world coordinates,
//	passes through the selected point, (x, y, -1/2, 1) in world coordinates,
//	and continues into the scene.
typedef struct
{
	//	Parameterize the ray as
	//
	//			P₀ + t(P₁ - P₀)
	//
	double	p0[4],	//	in tiling coordinates
			p1[4];	//	in tiling coordinates
	
	//	tMin marks where the ray enters the frame cell.
	//	tMax marks where the ray leaves the frame cell.
	double	tMin,
			tMax;

} HitTestRay3D;


//	A reset animation may be either 2D (spinning the square UIView)
//	or 3D (spinning the fundamental cube in ViewBasicLarge
//	or fading in and out in ViewRepeating), and may have
//	one of four possible purposes.
typedef enum
{
	ResetPlain,
	ResetWithNewGame,
	ResetWithNewTopology,
	ResetWithNewDifficultyLevel
} ResetPurpose;



//	for Intro

#define NUM_2D_INTRO_SPRITES	4

typedef struct
{
	Placement2D		itsSpritePlacement[NUM_2D_INTRO_SPRITES];
	unsigned int	itsDragSprite;
} IntroData2D;


//	for Tic-Tac-Toe

//		2D and 3D

typedef enum
{
	PlayerNone = 0,	//	Start at 0 to permit use as array index.
	PlayerX,
	PlayerO
} TicTacToePlayer;

//		2D only

typedef struct
{
	TicTacToePlayer	itsBoard[3][3];
	bool			itsCellFlip[3][3];
	TicTacToePlayer	itsWhoseTurn;
	unsigned int	itsHitCellH,
					itsHitCellV;
	Placement2D		itsWinLine;
} TicTacToeData2D;

//		3D only

typedef struct
{
	unsigned int	itsCenterNode[3];
	signed int		itsDirection[3];	//	each component = -1, 0 or +1;
										//		can't all be zero
} WinLine;

#define TIC_TAC_TOE_3D_SIZE			3
#define TIC_TAC_TOE_3D_CELL_WIDTH	(1.0 / TIC_TAC_TOE_3D_SIZE)

typedef struct
{
	//	Terminology:  The tic-tac-toe board is divided into
	//	a TIC_TAC_TOE_3D_SIZE × TIC_TAC_TOE_3D_SIZE × TIC_TAC_TOE_3D_SIZE
	//	lattice of "cells", but we're already using the word "cell"
	//	to refer to the "game cell" and the "frame cell".
	//	To avoid confusion, think of each tic-tac-toe "cell" as a "node".
	//	This usage also provides some consistency with the maze terminology.
	
	TicTacToePlayer	itsBoard[TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE][TIC_TAC_TOE_3D_SIZE];
	TicTacToePlayer	itsWhoseTurn;
	
	//	During a content drag, keep track of which node was hit originally,
	//	and accept the move iff the drag ends over that same node.
	//	This gives the user a chance to change his/her mind,
	//	and also helps protect against accidentally making a move
	//	when trying to rotate or scroll.
	//
	//	When the computer makes its move, it also writes it
	//	into itsHitNode so it may use the same animation code.
	//
	unsigned int	itsHitNode[3];
	
	//	Keep a flag saying whether the drag is currently
	//	over the original node or not.  When no drag is in progress,
	//	keep this flag set to false, so the drawing code may rely
	//	on it for coloring unoccupied node markers.
	//
	//	When the computer makes its move, it also sets
	//	itsHitRemainsValid so it may use the same animation code.
	//
	bool			itsHitRemainsValid;
	
	//	When the user makes a move, itsHitIsByHuman is true.
	//	When the computer responds, itsHitIsByHuman is false.
	bool			itsHitIsByHuman;
	
	//	When the user finalizes a hit, expand the selected marker to full size
	//	while shrinking the unselected PlayerNone markers.
	double			itsSelectedMarkerSizeFactor,
					itsUnselectedMarkerSizeFactor,
					itsMarkerSizeFactor;
	
	//	When somebody wins, draw a tube through the winning three-in-a-row.
	WinLine			itsWinningThreeInARow;

} TicTacToeData3D;


//	for Gomoku

//	Play on a 6×6 board.
//
//		Note #1:  While it would be possible to implement Gomoku
//		on a bigger board, it would be a bad idea, for two reasons:
//
//			First, it seems that human players win by thinking
//			in terms of long sequences of forcing moves,
//			so our broad, constant-depth search would likely
//			be a poor player.
//
//			Second, and far more importantly, bigger boards
//			"feel the topology" less than smaller boards do,
//			so they'd be pedagogically less effective.
//
//		Note #2:  At the other extreme, a 5×5 board would
//		be a bad idea because instead of have five distinct
//		five-in-a-row passing through each point in each direction,
//		there'd be only one such line.  So it would be
//		a very different game, and also the code that works fine
//		for all values of GOMOKU_SIZE ≥ 6 would need to be
//		substantially revised for GOMOKU_SIZE = 5.
//
#define GOMOKU_SIZE	6

typedef enum
{
	GomokuGameInProgress,
	GomokuGameWin,
	GomokuGameDraw
} GomokuStatus;

//	GomokuPlayerNone, GomokuPlayerBlack and GomokuPlayerWhite
//	get used as array indices, so don't change their values.
typedef enum
{
	GomokuPlayerNone	= 0,
	GomokuPlayerBlack	= 1,
	GomokuPlayerWhite	= 2
} GomokuPlayer;

typedef struct
{
	//	Each five-in-a-row line keeps track of the number
	//	of empty intersections, the number of black stones
	//	and the number of white stones that it contains.
	//
	//	The three values are indexed using the GomokuPlayer enum.
	//
	unsigned int	itsNumStones[3];	//	number of empty, black and white intersections
	
} GomokuLine;

typedef struct
{
	GomokuPlayer	itsStone;	//	none, black or white
	
	//	Assuming GOMOKU_SIZE ≥ 6, each intersection
	//	sits on 20 distinct five-in-a-row lines
	//	(5 such lines in each of 4 directions).
	GomokuLine		*itsLines[20];
} GomokuIntersection;

typedef struct
{
	//	The intersections
	GomokuIntersection	itsIntersections[GOMOKU_SIZE][GOMOKU_SIZE];

	//	The five-in-a-row lines
	//
	//		itsFiveInARowLines[h][v][d]
	//			is the line with center (h,v) and direction d,
	//			where d ∈ {0,1,2,3} is an index
	//			into the array gGomokuDirections.
	//
	GomokuLine			itsFiveInARowLines[GOMOKU_SIZE][GOMOKU_SIZE][4];

	//	The total number of stones of each color (or blank) on the whole board.
	//	The three values are indexed using the GomokuPlayer enum.
	unsigned int		itsTotalNumStones[3];	//	number of empty, black and white intersections
	
	//	itsOccupancyTallies[b][w] tells how many five-in-a-row lines
	//	contain exactly
	//
	//		b black stones and
	//		w white stones,
	//
	//	(with each such line's remaining 5 - (b + w) intersections
	//	still empty).
	unsigned int		itsOccupancyTallies[6][6];

} GomokuBoard;

typedef struct
{
	GomokuBoard		itsBoard;
	GomokuPlayer	itsWhoseTurn;
	GomokuStatus	itsGameStatus;
	unsigned int	itsHitIntersectionH,
					itsHitIntersectionV;
	Placement2D		itsWinLine;

	bool			itsThreadThinkingFlag,		//	for separate thread to signal completion
					itsThreadAbortFlag;			//	for main thread to ask separate thread to terminate ASAP
	unsigned int	itsThreadBestMoveH,			//	for separate thread to post its output
					itsThreadBestMoveV;
} GomokuData2D;


//	for Maze

//		2D only

typedef enum
{
	WallWest = 0,	//	Start at 0 to permit use as array index.
	WallSouth,
	WallEast,
	WallNorth
} Maze2DWallIndex;

typedef struct
{
	bool			itsWalls[4];	//	indexed by Maze2DWallIndex
	unsigned int	itsIndex;		//	used temporarily during maze generation
} Maze2DCell;

typedef struct
{
	unsigned int	itsSize;
	bool			itsMazeIsValid;
	Maze2DCell		**itsCells;	//	 See TorusGames2DMaze.c for coordinate conventions.
	Placement2D		itsMousePlacement,
					itsCheesePlacement;
} MazeData2D;

//		3D only

#define MAX_MAZE_3D_SIZE	5

typedef struct
{
	//	Does an edge connect this node to the next node
	//	in the positive x, y or z direction?
	bool	itsOutbound[3];

	//	Does an edge come in from the previous node
	//	in the negative x, y or z direction?
	bool	itsInbound[3];

} Maze3DNodeEdges;

typedef struct
{
	Byte	p[3];	//	(h,v,d)
} Maze3DNodePosition;

typedef struct
{
	Maze3DNodePosition	itsStartingNode;
	Byte				itsDirection;	//	0, 1 or 2, meaning direction h, v or d
} Maze3DEdge;

typedef enum
{
	SliderAtNode,
	SliderOnOutboundEdge
} Maze3DSliderStatus;

typedef struct
{
	Maze3DSliderStatus	itsStatus;
	
	//	If itsStatus == SliderAtNode,
	//		itsNode tells the node.
	//
	//	If itsStatus == SliderOnOutboundEdge,
	//		itsNode tells the edge's base node.
	//
	Maze3DNodePosition	itsNode;
	
	//	If itsStatus == SliderAtNode,
	//		itsDirection and itsDistance are undefined and ignored.
	//
	//	If itsStatus == SliderOnOutboundEdge,
	//		itsDirection tells which axis the slider is displaced along, and
	//		itsDistance tells how far the slider is from itsNode,
	//		as a fraction of the distance to the next node.
	//
	Byte				itsDirection;	//	0, 1 or 2, corresponding to h, v or d
	double				itsDistance;	//	0 < itsDistance < 1

} Maze3DSlider;

typedef struct
{
	unsigned int		itsSize;
	Maze3DNodeEdges		itsEdges[MAX_MAZE_3D_SIZE][MAX_MAZE_3D_SIZE][MAX_MAZE_3D_SIZE];

	Maze3DSlider		itsSlider;	//	marks player's position in maze
	Maze3DNodePosition	itsGoalNode;

	//	Several images of the slider may be visible simultaneously.
	//	When the user drags the slider, record the placement
	//	(in world coordinates) of the game cell image
	//	that contains the slider image that the user selected.
	double				itsDragCellIntoWorld[4][4];
	
	//	Show the brief 3D Maze how-to-play instructions
	//	during the first maze only.
	bool				itsShowInstructionsFlag;

} MazeData3D;


//	for Crossword

#define MAX_CROSSWORD_SIZE			 7	//	Must be ≤ 10 to accommodate clue indexing scheme.
#define	MAX_CROSSWORD_CLUES			21
#ifdef USE_CUSTOM_KLEIN_BOTTLE_CROSSWORD_FOR_TALK
#warning Remove this #ifdef after Leipzig Summer School is over.
#define MAX_CROSSWORD_CLUE_LENGTH	128	//	some clues for Leipzig are very long
#else
#define MAX_CROSSWORD_CLUE_LENGTH	80
#endif

#define CROSSWORD_NO_CLUE	0xFF

//	Assemble a Hangul syllable as the user types the jamo.
typedef struct
{
	//	The syllable's constituent jamo are stored in unicode
	//	not as explicit leading (0x1100-0x1112), vowel (0x1161-0x1175)
	//	and trailing (0x11A8-0x11C2) jamo, but rather 
	//	as "Hangul compatibility jamo" (0x3131-0x3163), because
	//
	//		1.	the Korean Input Method passes compatibility jamo,
	//			with no distinction for leading or trailing consonants, and
	//
	//		2.	as the Crossword code assembles syllables,
	//			it may need to remove a trailing consonant from one syllable
	//			and re-assign it as the leading consonant in the next syllable,
	//			for example  질 + ㅣ  →  지리 .
	//
	Char16	itsLeadingConsonant,
			itsVowel,
			itsTrailingConsonant;
	
	//	The combination of the current (possibily incomplete) set of jamo
	//	gets assembled into a tentative Hangul syllable, which is stored
	//	not in this structure but rather in itsPendingCharacter (see below).
			
} HangulSyllable;

//	Initialize a HangulSyllable to all zeros.
#define EMPTY_HANGUL_SYLLABLE	((HangulSyllable) {0x0000, 0x0000, 0x0000})


typedef struct
{
	Char16			itsPreviousLanguageCode[3];	//	Two-letter language code, such as "en" or "ja", with terminating zero
	unsigned int	itsPuzzleSize,
					itsIndexTorus,
					itsIndexKlein;
	Char16			itsBoard[MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE],
					itsSolution[MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE];
	bool			itsBoardFlips[MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE],
					itsLeftwardWord[MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE];
	unsigned int	itsHotCellH,
					itsHotCellV;
	bool			itsHotCellFlip;
	Char16			itsHotDirection;	//	← → ↓
	bool			itsHotWord[MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE];
	Char16			itsPendingCharacter;	//	supports accents (e.g. ':' + 'u' = 'ü') and kana composition (e.g. 's' + 'u' = 'す')
	HangulSyllable	itsPendingHangulSyllable;
	Char16			itsCluesText[MAX_CROSSWORD_CLUES][MAX_CROSSWORD_CLUE_LENGTH];
	Byte			itsClueIndexAcross[MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE],
					itsClueIndexDown  [MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE];
} CrosswordData2D;


//	for Word Search

#define	MAX_WORDSEARCH_SIZE			12	//	Must be ≤ 15 to accommodate texture indexing scheme.
#define MAX_WORDSEARCH_NUM_WORDS	48
#define MAX_WORDSEARCH_WORD_LENGTH	26	//	not including terminating zero

typedef struct
{
	uint8_t	itsStartH,
			itsStartV;
	int8_t	itsDeltaH,
			itsDeltaV;
	float	itsColor[4];	//	premultiplied alpha; linear Extended-Range sRGB on iOS
} WordSearchLine;

typedef struct
{
	Char16			itsPreviousLanguageCode[3];	//	Two-letter language code, such as "en" or "ja", with terminating zero
	unsigned int	itsPuzzleSize,
					itsIndexTorus,
					itsIndexKlein;
	Char16			itsBoard[MAX_WORDSEARCH_SIZE][MAX_WORDSEARCH_SIZE];
	bool			itsBoardFlips[MAX_WORDSEARCH_SIZE][MAX_WORDSEARCH_SIZE];
	Char16			itsWords[MAX_WORDSEARCH_NUM_WORDS][MAX_WORDSEARCH_WORD_LENGTH + 1];
	bool			itsWordSelectionIsPending,
					itsWordSelectionBeganWithCurrentDrag;
	unsigned int	itsStartH,
					itsStartV;
	double			itsDeltaH,
					itsDeltaV;
	double			itsDragStartH,				//	In the touch-based interface, a move may
					itsDragStartV;				//		consist of several separate drags,
	bool			itsDragWasTap;				//		concluded with a simple tap.
	float			itsColor[4];	//	premultiplied alpha; linear Extended-Range sRGB on iOS
	unsigned int	itsNumLinesFound;
	WordSearchLine	itsLines[MAX_WORDSEARCH_NUM_WORDS];
} WordSearchData2D;


//	for Jigsaw

#define MAX_JIGSAW_SIZE	5

typedef enum
{
	TabFemale	= 0,
	TabMale		= 1
} TabGender;

#define OPPOSITE_GENDER(g)	((TabGender)!(unsigned int)(g))

#define NUM_TORUS_PUZZLE_IMAGES		11
#define NUM_KLEIN_PUZZLE_IMAGES		 4

typedef struct _JigsawPiece JigsawPiece;
struct _JigsawPiece
{
	//	Logical placement of piece in puzzle
	unsigned int	itsLogicalH,			//	 ∈ {0, 1, ..., PuzzleSize - 1}, runs left-to-right
					itsLogicalV;			//	 ∈ {0, 1, ..., PuzzleSize - 1}, runs bottom-to-top

	Placement2D		itsPlacement;			//	current placement on board
	JigsawPiece		*itsNeighborWest,		//	pointers to the neighbors
					*itsNeighborEast,		//		in the Klein bottle, west-east
					*itsNeighborSouth,		//		gets defined in the obvious way,
					*itsNeighborNorth;		//		relative to the standard fundamental domain
	bool			itsNeighborBondWest,	//	attached to neighboring pieces?
					itsNeighborBondEast,	//		(values must be 0 or 1 because we use them as array indices)
					itsNeighborBondSouth,
					itsNeighborBondNorth;
	TabGender		itsTabGenderWest,
					itsTabGenderEast,
					itsTabGenderSouth,
					itsTabGenderNorth;
	bool			itsClumpIsNonorientable,
                    itsTextureBordersNeedUpdate,	//  texture needs update to remove one or more borders?
					itsVisitedFlag,					//	for local use in managing a recursion
					itsReflectionFlag;				//	also for local use
};

typedef struct
{
	unsigned int	itsSize;
	JigsawPiece		itsPieces[MAX_JIGSAW_SIZE * MAX_JIGSAW_SIZE],
					*itsPieceOrder[MAX_JIGSAW_SIZE * MAX_JIGSAW_SIZE];

	double			itsPuzzlePieceCollageFirstCenter,	//	For locating piece images in the puzzle image texture,
					itsPuzzlePieceCollageStride;		//		in [0.0, 2.0] coordinates.

	unsigned int	itsCurrentTorusPuzzle,
					itsCurrentKleinPuzzle;
	JigsawPiece		*itsDragPiece;
} JigsawData2D;


//	for Chess

//	A single unsigned integer describes the contents
//	of each square on the chessboard via the following flags:

#define CHESS_EMPTY_SQUARE	0x00000000

#define CHESS_PIECE_MASK	0x00000007
#define CHESS_KING			0x00000001
#define CHESS_QUEEN			0x00000002
#define CHESS_BISHOP		0x00000003
#define CHESS_KNIGHT		0x00000004
#define CHESS_ROOK			0x00000005
#define CHESS_PAWN			0x00000006

#define CHESS_COLOR_MASK	0x00000008
#define CHESS_WHITE			0x00000000
#define CHESS_BLACK			0x00000008

#define CHESS_PARITY_MASK	0x00000010
#define CHESS_PLAIN			0x00000000
#define CHESS_REFLECTED		0x00000010

typedef struct
{
	unsigned int	itsStartH,
					itsStartV;
	signed int		itsDeltaH,
					itsDeltaV;
	unsigned int	itsFinalH,
					itsFinalV;
} ChessMove;

typedef struct
{
	unsigned int	itsSquares[8][8];			//	[ h (rightward 0..7) ][ v (upward 0..7) ]
	unsigned int	itsWhoseTurn;				//	CHESS_WHITE or CHESS_BLACK

	bool			itsUserMoveIsPending,
					itsComputerMoveIsPending,	//	Is the app dragging a piece for the computer's move?
					itsUserMoveBeganWithCurrentDrag;
	unsigned int	itsMoveStartH,
					itsMoveStartV;
	Placement2D		itsMovePlacement;
	double			itsDragStartH,				//	In the touch-based interface, a move may
					itsDragStartV;				//		consist of several separate drags,
	bool			itsDragWasTap;				//		concluded with a simple tap.

	bool			itsThreadThinkingFlag,		//	for separate thread to signal completion
					itsThreadAbortFlag;			//	for main thread to ask separate thread to terminate ASAP
	ChessMove		itsThreadBestMove;			//	for separate thread to post its output

	ChessMove		itsCheckThreat;				//	Will be displayed iff itsDeltaH != 0 || itsDeltaV != 0.
	bool			itsCheckmateFlag;			//	When game is over, flag will be true for checkmate or false for stalemate.
} ChessData2D;


//	for Pool

//	Ball  0   is the cue ball.
//	Balls 1-3 are the solid colors.
//	Ball  4   is black (in effect an "eight ball").
//	Balls 5-7 are the striped colors.
#define NUM_POOL_BALLS	8

//	Two players compete in each pool game.
//	If the computer plays, it typically plays for Player B
//	(exception:  the user may switch from human vs. human
//	to human vs. computer mode at any time, in which case
//	the computer might end up playing for Player A).
typedef enum
{
	PoolPlayerA = 0,	//	These values permit toggling with the ! operator.
	PoolPlayerB = 1
} PoolPlayer;

//	The first player to sink a ball gets that type (solids or stripes).
//	Once all balls of that types get sunk, the player shoots
//	for the black ball (the 4-ball).
typedef enum
{
	PoolUndecided,
	PoolSolids,
	PoolStripes,
	PoolBlack
} PoolGoal;

typedef enum
{
	PoolCueNone,		//	Pool cue is not visible
						//		(for example, while balls are rolling)
	PoolCueVisible,		//	Pool cue is visible, but not under direct user control
						//		(for example, while user is scrolling the board)
	PoolCueActive		//	Pool cue is under direct user control
						//		(for example, while user is lining up his/her shot)
} PoolCueStatus;

typedef struct
{
	PoolPlayer		itsWhoseTurn;		//	Is it PoolPlayerA's turn or PoolPlayerB's turn?
	PoolGoal		itsPlayerAGoal,		//	Who's shooting for what?
					itsPlayerBGoal;
	bool			itsWinFlag,			//	Has the current player won?
					itsLoseFlag;		//	Has the current player lost?
	PoolPlayer		itsComputerPlayer;	//	If itsHumanVsComputer is true, is the computer playing for PoolPlayerA or PoolPlayerB?
	bool			itsBallInPlay[NUM_POOL_BALLS];		//	Is the given ball on the table?  If so...
	Placement2D		itsBallPlacement[NUM_POOL_BALLS];	//	...where is it?
	PoolCueStatus	itsCueStatus;
	bool			itsCueBecameActiveWithCurrentDrag;
	Placement2D		itsCueHotSpot;		//	The middle part of the cue stick, where the hand grabs it.
	double			itsDragStartH,		//	In the touch-based interface, a cue stick movement may
					itsDragStartV;		//		consist of several separate drags,
	bool			itsDragWasTap;		//		concluded with a simple tap.
	double			itsBallVelocityH[NUM_POOL_BALLS],	//	relative to ball's local coordinates, in units
					itsBallVelocityV[NUM_POOL_BALLS];	//		of distance per second
	bool			itsPassiveScratchFlag,	//	If the user sinks no balls, that's a passive scratch.
					itsActiveScratchFlag;	//	If the user sinks an illegal ball, that's an active scratch.
} PoolData2D;


//	for Apples

//	A single AppleStatus byte records the status of each square
//	in the Apples2D games.
//	The four low-order bits (bits 0 through 3) record how many
//		of the square's eight neighbors contains a worm.
//	Bit 4 records whether the apple itself has a worm.
//	Bit 5 records whether the apple has been exposed or not.
//	Bit 6 records whether user has marked the cell.
//	Bit 7 records whether the cell is drawn with a flip or not.
//
//	Define masks as unsigned integers rather than signed integers,
//	so that (itsBoard[h][v] & APPLE_NBR_COUNT_MASK) will be unsigned.
#define APPLE_NBR_COUNT_MASK	0x0Fu
#define APPLE_HAS_WORM_MASK		0x10u
#define APPLE_EXPOSED_MASK		0x20u
#define APPLE_MARKED_MASK		0x40u
#define APPLE_FLIPPED_MASK		0x80u
typedef Byte AppleStatus;

//	What is the biggest allowable board size?
#define MAX_APPLE_BOARD_SIZE	64

//	The numbers look best at 256×256 resolution,
//	but soak up large amounts of both video RAM and regular RAM.
//	To conserve RAM, let's render the numbers at 128×128,
//	which looks almost as good.
#define APPLES_CHARACTER_TEXTURE_SIZE	128

typedef struct
{
	unsigned int	itsBoardSize;
	AppleStatus		itsBoard[MAX_APPLE_BOARD_SIZE][MAX_APPLE_BOARD_SIZE];
	unsigned int	itsHitCellH,
					itsHitCellV;
	bool			itsRightClickFlag;
	bool			itsShowBriefInstructionsFlag;
	unsigned int	itsShowBriefInstructionsMarkedWormCount;
} ApplesData2D;


//	All platform-independent data about the space and
//	how it's displayed live in the ModelData structure.
struct ModelData
{
	//	When the platform-independent code modifies the ModelData
	//	it increments itsChangeCount as a signal to the platform-dependent code
	//	that it should redraw the view at the next opportunity.
	//
	//		Note:  It's OK to have two or more views of the same ModelData.
	//		Each view keeps its own private variable telling the change count
	//		when the view was last redrawn.
	//
	uint64_t			itsChangeCount;		//	wraparound, while unlikely, would be harmless

	//	In a 2D game, our basic drawing code will draw
	//	into a square fundamental domain with x and y coordinates
	//	each running from -0.5 to +0.5.  The window has the
	//	same dimensions as the fundamental domain, and initially
	//	the two coincide.  However, the user may choose to scroll
	//	the fundamental domain relative to the window.
	//	Keep track of the offset.
	//	We're still in the same coordinate system, so the offset
	//	will also have x and y components in the range -0.5 to +0.5.
	//	Also keep track of whether the image is reflected,
	//	as can happen in the Klein bottle.
	Offset2D			itsOffset;

	//	Which game?
	GameType			itsGame;

	//	Which topology?
	TopologyType		itsTopology;

	//	Which view type?
	ViewType			itsViewType;
	double				its2DViewMagFactor;	//	∈ [1/3, 1] where
											//	1/3 corresponds to ViewBasicSmall (in 2D Games)
											//	 1  corresponds to ViewBasicLarge (in 2D Games)
	
	//	Animate transitions between ViewBasicSmall and ViewBasicLarge.
	ViewType			itsDesiredViewType;
	double				itsDesired2DViewMagFactor;	//	∈ [1/3, 1]

	//	Play human vs. computer in tic-tac-toe, chess and pool?
	bool				itsHumanVsComputer;

	//	How difficult are maze, jigsaw and chess?
	//
	//		0 = easy, 1 = medium, 2 = hard, 3 = extra hard
	//
	unsigned int		itsDifficultyLevel;

	//	Draw the glide reflection axes in Klein bottle mode?
	bool				itsShowGlideAxes;

	//	Has someone won yet?
	bool				itsGameIsOver,
						itsFlashFlag;

	//	Keep track of the most recent mouse/touch began event,
	//	so we can know a drag's total elapsed time.
	double				itsDragBeganTime;		//	in seconds, since some arbitrary "time 0"

	//	Keep track of the most recent mouse/touch began or moved event,
	//	so we can estimate a velocity.
	double				itsPreviousDragTime;	//	in seconds, since some arbitrary "time 0"
	
	//	Keep track of the previous point while dragging.
	double				itsPreviousDragPoint[2];	//	in [-0.5, +0.5] view coordinates
	
	//	In a 3D game, is the current drag for scrolling, rotating or content?
	//
	//		Note:	iOS reports UIGestureRecognizerStateBegan *before* it calls
	//				-touchesCancelled:withEvent: to cancel manual handling of the touch sequence.
	//				So we need separate variables its3DDragType and its3DGestureDragType
	//				to avoid conflicts.
	//
	DragType3D			its3DDragType,			//	for "ordinary" drags
						its3DGestureDragType;	//	for gesture-based drags
	
	//	In a 3D game, as the user rotates the fundamental cube,
	//	keep track of its rotational velocity for use
	//	after the user lifts his/her finger, for coasting.
	Velocity			its3DRotationalVelocity;

	//	In a 3D game, pick an axis at the beginning of a scroll,
	//	and stick with it until the scroll ends.
	bool				its3DScrollAxisIsKnown;
	unsigned int		its3DScrollAxis;	//	0, 1 or 2 for x, y or z
	double				its3DProjectedScrollAxis[2],
						its3DScrollFactor,	//	= (distance along axis) / (scroll distance)
						its3DScrollSpeed;	//	used only for coasting
	
	//	In a 3D game, at the end of a scroll, if the tiling's center
	//	has any of its three coordinates close to a grid value,
	//	animate it to exactly that grid value.
	double				its3DTranslationSnapBegin[4],
						its3DTranslationSnapEnd[4],
						its3DTranslationSnapDuration;
	
	//	In a 3D game, animatate the frame cell to the nearest axis-aligned position
	//
	//	  - when switching from ViewBasicLarge to ViewRepeating, or
	//
	//	  - when the frame cell almost aligns with the axes at the end of a rotation.
	//
	Isometry			its3DRotationSnapBegin,
						its3DRotationSnapEnd;
	double				its3DRotationSnapDuration;
	bool				its3DRotationSnapChangeToTiling;	//	change to ViewRepeating upon completion?

	//	In a 2D game, keep track of the hand cursor.
	//	On an iOS device, its2DHandStatus may take the values
	//	{HandNone, HandScroll, HandDrag}, but never HandFree.
	HandStatus2D		its2DHandStatus;
	Placement2D			its2DHandPlacement;		//	cursor placement in fundamental domain, [-0.5, +0.5] coordinates
	bool				its2DDragIsReversed;	//	Do the dragged object's local coordinates agree or disagree
												//	with the hand's local coordinates?  Individual Games may
												//	choose to use this field or ignore it.
	double				its2DDragHistory[DRAG_HISTORY_DEPTH][2];	//	Only GameIntro and GameMaze use this field.

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	//	In a 2D game, when the left mouse button goes down,
	//	keep track of whether it's a double-click or not,
	//	then respond appropriately when the mouse button comes up.
	bool				its2DPreviousClickWasDoubleClickEligible,
						its2DDoubleClickFlag;
	Placement2D			its2DPrevClickPlacement;	//	for judging double-clicks
#endif

	//	Let the board coast gently to a stop
	//	after the user lifts his/her finger or mouse button.
	CoastingType		itsCoastingStatus;
	
	//	Coasting2DTranslation	uses its2DCoastingVelocity.
	//	Coasting3DTranslation	uses the its3DScroll* set of variables, defined above.
	//	Coasting3DRotation		uses its3DRotationalVelocity, defined above.
	double				its2DCoastingVelocity[2];
	
	//	For the most part, the games don't care whether the various views
	//	(main game view, possible keyboard view, possible message view)
	//	are arranged in a portrait layout or a landscape layout.
	//	The except is the Chinese Word Search, in which the poems
	//	must be formatted as a narrow column in a landscape layout.
	bool				itsChinesePoemsWantColumnFormatting;

	//	In a 3D game, a sequence of maps
	//
	//		object → game cell → tiling → frame cell → world
	//
	//	takes each object into world space.
	//
	//	Game Cell
	//		The game cell is a cube with x, y and z coordinates
	//		running from -0.5 to +0.5.  Along with the appropriate 
	//		gluings on its boundary, this cell defines the space 
	//		in which the game is played.
	//
	//	Tiling
	//		The game cell maps into tiling space via an element
	//		of the tiling group.  The identity element of the tiling group
	//		naturally gets represented by the identity matrix,
	//		so tiling coordinates agree with game cell coordinates
	//		for the tile centered at (0,0,0).  Repeating images
	//		of the game cell fill out the tiling as far as desired.
	//	
	//	Frame Cell
	//		The frame cell defines the visible walls that the user sees.
	//		It's a cube with x, y and z coordinates running from -0.5 to +0.5.
	//		Initially its near face coincides exactly with the viewport,
	//		but then moves away as the user rotates the frame cell.
	//		The tiling maps into frame cell coordinates, with the tiling's origin
	//		always somewhere within the frame cell proper.
	//		If, during scrolling, the tiling's center goes beyond the frame cell,
	//		we apply an element of the tiling group to bring it back.
	//
	//	World Space
	//		Finally, the frame cell maps into world space.
	//		The frame cell origin always maps to the world space origin.
	//		During a rotation the frame cell may map into world space
	//		via any rotation of SO(3).
	//
	//	While scrolling gets realized in the map
	//
	//			tiling → frame cell
	//
	//	a rotation gets realized in the map
	//
	//			frame cell → world
	//
	//	The latter carries the walls along with it,
	//	while the former does not.
	//
	//	During scrolling, mapping 2D finger motions
	//	(on a touchpad or touchscreen) to 3D translations
	//	is tricky.  To resolve the ambiguity, Torus Games
	//	scrolls only in the directions of the coordinate axes,
	//	i.e. in the direction of only one axis at a time.
	//	When the user begins each scroll, the program selects
	//	the axis closest to the initial direction of the user's
	//	finger or mouse motion.
	//
	//		Design note:  Store its3DFrameCellIntoWorld
	//		not as a 3×3 matrix, but rather as a spin vector
	//		(better known as a "versor" here in the case of SO(3))
	//		to facilitate the snap-to-axis-aligned-orientation feature.
	//
	double				its3DTilingIntoFrameCell[4][4];
	Isometry			its3DFrameCellIntoWorld;	//	∈ Spin(3) = double cover of SO(3)

	//	For a 3D game, the aperture in each face of the fundamental cube
	//	may be fully closed (0.0), fully open (1.0), or anywhere in between.
	//	User commands change its3DDesiredAperture immediately,
	//	but its3DCurrentAperture catches up only gradually,
	//	providing a smooth animation.
	double				its3DDesiredAperture,
						its3DCurrentAperture;	
	
	
	//	If a 3D spinning cube reset animation is in progress, what kind is it?
	//	And what is the new topology or difficulty level?
	//
	//	Note:
	//	its3DResetPurpose excludes ResetWithNewGame, for two reasons:
	//
	//	  -	so the UI may present all game transitions the same way,
	//		regardless of whether the old and new games are 2D, 3D or mixed,
	//
	//	  - to keep the code simple, with no need for the platform-independent
	//		reset function to make messy calls back up to the UI to update toolbars, etc.
	//	
	ResetPurpose		its3DResetPurpose;
	unsigned int		its3DResetNewValue;
	
	//	For a 3D game in fundamental domain mode,
	//	what was the frame cell's orientation
	//	at the start of the reset game animation?
	Isometry			its3DResetGameStartingOrientation;

	//	For a 3D game in fundamental domain mode,
	//	what axis should the reset game animation
	//	rotate the frame cell about?
	double				its3DResetGameRotationAxis[3];	//	(x,y,z) in world coordinates

	//	For a 3D game in fundamental domain mode,
	//	during the reset game animation, what scale factor
	//	should be applied to the fundamental domain?
	double				its3DResetGameScaleFactor;

	//	For a 3D game in fundamental domain mode,
	//	during the reset game animation, which walls should stay opaque?
	//	The indexing order is that of gWallIntoFrameCellPlacements.
	bool				its3DResetGameOpaqueWalls[NUM_FRAME_WALLS];
		
	//	For a 3D game in tiling mode, during the reset game animation,
	//	what is the current fog shift as a fraction of the total fog range?
	double				its3DResetGameTilingFogShift;
	
	//	A drag (with mouse or finger) may trigger a message.
	//	Currently only Gomoku2D uses this feature (to ask the user
	//	to place stones on intersections, not in squares).
	TitledErrorMessage	itsDragMessage;
	
	//	Keep track of what simulation (typically none) is running during idle time.
	//	If one is running, keep track of how long it's been running.
	SimulationType		itsSimulationStatus;
	double				itsSimulationElapsedTime,	//	total elapsed time in seconds
						itsSimulationDeltaTime;		//	time since previous update, in seconds
	
	//	Long computation support
	bool				itsProgressMeterIsActive;
	double				itsProgressMeter;	//	runs from 0.0 (start) to 1.0 (finish)

	//	Accumulated drag when deciding on the 3D scrolling direction
	//	or the initial direction of the slider in the 3D Maze.
	double				itsAccumulated3DDragMotion[2];

	//	Game-dependent functions
	//
	//	When a game starts up it sets these function pointers
	//	and when it shuts down it restores them to NULL.
	//
	//	If you prefer object-oriented language, you may think of each game
	//	as a subclass of a generic game class.  In effect each game
	//	implements methods that the base class defines only abstractly,
	//	only here we do it in C rather than C++.
	//
	//	If a game doesn't need to implement a particular method,
	//	it may instead set the corresonding pointer to NULL.
	//
	void				(*itsGameShutDown					)(ModelData *md);
	void				(*itsGameReset						)(ModelData *md);
	void				(*itsGameHumanVsComputerChanged		)(ModelData *md);
	void				(*itsGame2DHandMoved				)(ModelData *md);
	bool				(*itsGame2DDragBegin				)(ModelData *md, bool aRightClick);
	void				(*itsGame2DDragObject				)(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);
	void				(*itsGame2DDragEnd					)(ModelData *md, double aDragDuration, bool aTouchSequenceWasCancelled);
	bool				(*itsGame3DDragBegin				)(ModelData *md, HitTestRay3D *aRay);
	void				(*itsGame3DDragObject				)(ModelData *md, HitTestRay3D *aRay, double aMotion[2]);
	void				(*itsGame3DDragEnd					)(ModelData *md, bool aTouchSequenceWasCancelled);
	unsigned int		(*itsGame3DGridSize					)(ModelData *md);
	TitledErrorMessage	*(*itsGameCharacterInput			)(ModelData *md, Char16 aCharacter);
	void				(*itsGameSimulationUpdate			)(ModelData *md);
	void				(*itsGameRefreshMessage				)(ModelData *md);

	//	Game-specific data
	//
	//	A given window supports only one game at a time, 
	//	so we may store the game-specific data in a union.
	//	(I would have preferred an unnamed union, but
	//	standard C does not officially support them.)
	union
	{
		IntroData2D			Intro2D;
		TicTacToeData2D		TicTacToe2D;
		GomokuData2D		Gomoku2D;
		MazeData2D			Maze2D;
		CrosswordData2D		Crossword2D;
		WordSearchData2D	WordSearch2D;
		JigsawData2D		Jigsaw2D;
		ChessData2D			Chess2D;
		PoolData2D			Pool2D;
		ApplesData2D		Apples2D;

		TicTacToeData3D		TicTacToe3D;
		MazeData3D			Maze3D;

	} itsGameOf;
};


//	Vertex data formats

//		for ordinary 2D vertices with texture coordinates
typedef struct
{
	simd_float2	pos,	//	position (x,y)
				tex;	//	texture coordinates (u,v)
} TorusGames2DVertexData;

//		for "offsettable" 2D vertices with texture coordinates,
//			for use in lines-with-endcaps
typedef struct
{
	simd_float2	pos,	//	position (x,y)
				tex;	//	texture coordinates (u,v)
	float		off;	//	∈ {0.0, 1.0} according to whether the vertex
						//		shouldn't or should be offset
} TorusGames2DOffsetableVertexData;

//		for 3D vertices for walls
typedef struct
{
	simd_float3	pos;	//	position (x,y,z) of outer vertex, in frame cell coordinates
	simd_float2	tex,	//	texture coordinates (u,v) of outer vertex (for algorithmic textures)
				wgt;	//	relative weights (s,t) of outer and inner vertices
						//		the plain wall uses only outer vertices
						//		the wall-with-aperture uses inner and outer both
} TorusGames3DWallVertexData;

//		for 3D vertices for content
typedef struct
{
	simd_float3		pos;	//	position (x,y,z), in the object's local [-0.5, +0.5] coordinates
	faux_simd_half3	nor;	//	normal vector (nx, ny, nz)
} TorusGames3DPolyhedronVertexData;


//	Polyhedron placement

typedef struct
{
	//	The shader applies itsDilation only to vertices, never to normal vectors.
	double	itsDilation[3];					//	scale factor in x, y and z directions
	
	//	The shader applies itsIsometricPlacement to normal vectors as well as vertices.
	double	itsIsometricPlacement[4][4];	//	reflects, rotates and translates as desired
	
	//	The shader applies itsExtraClippingCovector only in 3D Tic-Tac-Toe
	//	and even there only for cross sections of win-line tubes that intersect
	//	a frame cell wall obliquely.  Because the win-line tube has a finite length,
	//	sometimes its cross section is a clipped ellipse rather than a full ellipse.
	//	The shader evaluates the covector in frame cell coordinates
	//	and clips to the region where the covector times a vertex
	//	evaluates to a result in the range [-1.0, +1.0].
	//
	//	For polyhedra other than elliptical cross sections of Tic-Tac-Toe win-line tubes,
	//	this field should be set to the zero vector for tidiness, but ultimately
	//	it doesn't get used.
	//
	double	itsExtraClippingCovector[4];	//	to be evaluated on a point (x,y,z,1) in frame cell coordinates

} TorusGames3DPolyhedronPlacementAsCArrays;


//	Platform-independent constants and global variables

#define NUM_SQUARE_VERTICES				 4
#define NUM_LINE_WITH_ENDCAPS_VERTICES	 8
#define NUM_WALL_VERTICES				 4
#define NUM_WALL_WITH_APERTURE_VERTICES	10
#define NUM_CUBE_VERTICES				(6 * 4)
#define NUM_CUBE_FACETS					(6 * 2)	//	two triangles per face
#define NUM_CUBE_SKELETON_VERTICES		(6 * 8)
#define NUM_CUBE_SKELETON_FACETS		(6 * 8)

//	A ball may be realized as a mesh at various levels of refinement.
//
//		Refinement level 0 is a plain octahedron (8 faces).
//			The reasons for starting with an octahedron rather than an icosahedron are that
//				- the cross-sections of a subdivided octahedron fit perfectly
//					with axis-aligned connecting tubes having 2^(2 + n) vertices at each end,
//				- the Maze may draw only a half or a quarter of a ball where appropriate,
//					for a higher frame rate (especially in ViewRepeating).
//
//		Refinement level 1 is obtained by subdividing each of the octahedron's faces
//			into four smaller faces in the natural way, for a total of 4*8 = 32 faces.
//
//		Refinement level 2 is obtained by subdividing the level 1 faces,
//			for a total of 4*32 = 128 faces.
//
//		Refinement level 3 is obtained by subdividing the level 2 faces,
//			for a total of 4*128 = 512 faces.
//
//		etc.
//
//	The present implementation uses MAX_BALL_REFINEMENT_LEVEL = 4.
//
//		For ViewBasicLarge we use refinement level 4, which looks great
//			for the 3D Tic-Tac-Toe marker and the 3D Maze slider,
//			and easily runs at 60 fps on my iPad and iPod Touch
//			(using only about 40% of each frame period, in fact).
//
//		For ViewRepeating, however, refinement level 4 is too slow (~15 fps)
//			so we use refinement level 3 instead, which just barely runs at 60 fps.
//
#define MAX_BALL_REFINEMENT_LEVEL		4

//	The number of faces quadruples from one refinement level to the next.
#define MAX_REFINED_BALL_FACETS			(8 << (2 * MAX_BALL_REFINEMENT_LEVEL))

//	Each face sees three edges, but each edge gets seen twice.
#define MAX_REFINED_BALL_EDGES			(3 * MAX_REFINED_BALL_FACETS / 2)

//	Each face sees three vertices, and -- except for the six original vertices --
//	each vertex gets seen six times.  To compensate for the six original vertices,
//	each of which gets seen only four times, but must add in a correction factor
//	of 6*(6 - 4) = 12 "missing" vertex sightings.
#define MAX_REFINED_BALL_VERTICES		((3 * MAX_REFINED_BALL_FACETS + 12) / 6)

//	To summarize the effect of the preceding #definitions:
//
//		n	  v	  e	  f
//		----------------
//		0	  6	 12	  8
//		1	 18	 48	 32
//		2	 66	192	128
//		3	258	768	512
//		…	  …	  …	  …
//

//	The tube mesh is designed to fit exactly with the ball mesh,
//	to avoid any possibility that the player might "see through a gap".
#define MAX_TUBE_REFINEMENT_LEVEL		MAX_BALL_REFINEMENT_LEVEL

//	The number of faces doubles from one refinement level to the next.
#define MAX_REFINED_TUBE_FACETS			(8 << MAX_TUBE_REFINEMENT_LEVEL)

//	The number of vertices also doubles from one refinement level to the next.
#define MAX_REFINED_TUBE_VERTICES		(8 << MAX_TUBE_REFINEMENT_LEVEL)

//	A square has 4 vertices.
#define NUM_SQUARE_SLICE_VERTICES		 4

//	Match the slice refinement to the ball refinement,
//	so that when a slice passes through a ball's midpoint
//	the meshes match up and "no light can slip through".
#define MAX_SLICE_REFINEMENT_LEVEL		MAX_BALL_REFINEMENT_LEVEL
#define MAX_NUM_CIRCULAR_SLICE_VERTICES	(4 << MAX_SLICE_REFINEMENT_LEVEL)

#define NUM_AVAILABLE_WALL_COLORS		(NUM_FRAME_WALLS/2)

//	Tiling parameters
//
//	For the repeating view, the method
//
//		-write3DCoveringTransformationsIntoBuffersPlain:reflecting:modelData:
//
//	considers all the game cell's translates
//	from -TILING_SIZE_FOR_REPEATING_VIEW to +TILING_SIZE_FOR_REPEATING_VIEW
//	in each coordinate, and then culls to accept only those translates whose content
//	may intersect both the view frustum and a "clipping sphere" centered at the camera at (0,0,-1).
//	See comment immediately below for an explanation of the clipping sphere's radius.
//
#define TILING_SIZE_FOR_REPEATING_VIEW		3
#define CUBE_SIZE_FOR_REPEATING_VIEW		(TILING_SIZE_FOR_REPEATING_VIEW + 1 + TILING_SIZE_FOR_REPEATING_VIEW)
#define TOTAL_CUBE_CELLS_FOR_REPEATING_VIEW	(CUBE_SIZE_FOR_REPEATING_VIEW * CUBE_SIZE_FOR_REPEATING_VIEW * CUBE_SIZE_FOR_REPEATING_VIEW);

//	In the repeating view, the GPU vertex function will clip all game content
//	to a large "clipping sphere" centered at the camera at (0,0,-1).
//	The clipping sphere's radius should enclose as much of the tiling as possible,
//	without enclosing any "missing content".  For example,
//	if TILING_SIZE_FOR_REPEATING_VIEW == 3, then the nearest "missing content"
//	will be based in the translated game cell centered at (x,y,z) = (0,0,4)
//	in tiling coordinates.  However,
//
//		because the cell's contents may extend up to 0.5 units
//			beyond the cell's nominal boundary,
//		the nearest missing content may sit
//			just beyond (0,0,3) in tiling coordinates,
//	and
//		because the frame cell's placement in world space
//			may vary by up to 0.5 units in each direction,
//		the nearest missing content may sit
//			just beyond (0, 0, 2.5) in world coordinates.
//
//	So, in this example, we'd wouldn't want the clipping sphere
//	to extend beyond the point (0, 0, 2.5).
//	Because the camera sits at (0, 0, -1) in world coordinates,
//	this means that, in this example, we can safely set
//	the clipping sphere's radius to 3.5, but no larger.
//
#define CLIPPING_RADIUS_FOR_REPEATING_VIEW	(TILING_SIZE_FOR_REPEATING_VIEW + 0.5)


extern const Char16			gLanguages[][3];	//	for example {u"de", u"en", … , u"zs", u"zt"}
extern const unsigned int	gNumLanguages;


//	Matte color
extern const float	gMatteColorGammaP3[3];	//	in gamma-encoded Display P3 coordinates

//	Graphical primitives

extern const TorusGames2DVertexData				gSquareVertices[NUM_SQUARE_VERTICES];
extern const TorusGames2DOffsetableVertexData	gLineWithEndcapsOffsettableVertices[NUM_LINE_WITH_ENDCAPS_VERTICES];

extern const TorusGames3DWallVertexData			gWallVertices[NUM_WALL_VERTICES];
extern const TorusGames3DWallVertexData			gWallWithApertureVertices[NUM_WALL_WITH_APERTURE_VERTICES];

extern const TorusGames3DPolyhedronVertexData	gCubeVertices[NUM_CUBE_VERTICES];
extern const unsigned int						gCubeNumFacets;
extern const unsigned short						gCubeFacets[NUM_CUBE_FACETS][3];

extern const TorusGames3DPolyhedronVertexData	gCubeSkeletonVertices[NUM_CUBE_SKELETON_VERTICES];
extern const unsigned int						gCubeSkeletonNumFacets;
extern const unsigned short						gCubeSkeletonFacets[NUM_CUBE_SKELETON_FACETS][3];

//	Allocate space for each possible ball resolution from 0 up through MAX_BALL_REFINEMENT_LEVEL.
//	To keep the array definitions simple, allocate the same amount of space for each level,
//	even though the full amount will be needed only at the highest level.  The amount of "wasted" memory
//	is modest ( < 50 kb) and is a small price to avoid messing with different-size rows.
//
extern unsigned int								gBallNumVertices[MAX_BALL_REFINEMENT_LEVEL + 1];
extern TorusGames3DPolyhedronVertexData			gBallVertices[MAX_BALL_REFINEMENT_LEVEL + 1][MAX_REFINED_BALL_VERTICES];
extern unsigned int								gBallNumFacets[MAX_BALL_REFINEMENT_LEVEL + 1];
extern unsigned short							gBallFacets[MAX_BALL_REFINEMENT_LEVEL + 1][MAX_REFINED_BALL_FACETS][3];

//	Allocate space for each possible tube resolution from 0 up through MAX_TUBE_REFINEMENT_LEVEL.
//	Allocate the same amount of space for each level, for the same reason given for the ball arrays immediately above.
//
extern unsigned int								gTubeNumVertices[MAX_TUBE_REFINEMENT_LEVEL + 1];
extern TorusGames3DPolyhedronVertexData			gTubeVertices[MAX_TUBE_REFINEMENT_LEVEL + 1][MAX_REFINED_TUBE_VERTICES];
extern unsigned int								gTubeNumFacets[MAX_TUBE_REFINEMENT_LEVEL + 1];
extern unsigned short							gTubeFacets[MAX_TUBE_REFINEMENT_LEVEL + 1][MAX_REFINED_TUBE_FACETS][3];

extern const TorusGames3DPolyhedronVertexData	gSquareSliceVertices[NUM_SQUARE_SLICE_VERTICES];

//	Allocate space for each possible circular slice resolution from 0 up through MAX_SLICE_REFINEMENT_LEVEL.
//	Allocate the same amount of space for each level, for the same reason given for the ball arrays above.
//
extern unsigned int								gCircularSliceNumVertices[MAX_SLICE_REFINEMENT_LEVEL + 1];
extern TorusGames3DPolyhedronVertexData			gCircularSliceVertices[MAX_SLICE_REFINEMENT_LEVEL + 1][MAX_NUM_CIRCULAR_SLICE_VERTICES];

//	Frame cell walls
extern const double	gWallIntoFrameCellPlacements[NUM_FRAME_WALLS][4][4];
extern const float	gWallColors[NUM_AVAILABLE_WALL_COLORS][4];

//	Dummy placements for unneeded sprites.
extern const Placement2D	gUnusedPlacement,	//	for completely unused sprites
							gZeroPlacement;		//	for suppressed sprites


//	Platform-dependent global functions

//	in TorusGames*Callbacks.m
extern void		SetTorusGamesStatusMessage(const Char16 *aText, ColorP3Linear aColorP3Linear, GameType aGame);

//	Platform-independent global functions

//	in TorusGamesInit.c
extern void			ClearFunctionPointers(ModelData *md);

//	in TorusGamesProjection.c
extern void			MakeProjectionMatrix3D(ModelData *md, double aProjectionMatrix[4][4]);

//	in TorusGamesGraphics.c
extern unsigned int	GetNum2DBackgroundTextureRepetitions(ModelData *md);
extern void			Get2DKleinAxisColors(ModelData *md, float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DSprites(ModelData *md);
extern void			Get2DSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);
extern void			Make3DFogDistances(ModelData *md, __fp16 *aFogEnd, __fp16 *aFogScale);
extern double		*Make3DTexCoordShift(double a3DTilingIntoFrameCell[4][4], unsigned int aWallIndex, double aTexCoordShift[2]);
extern unsigned int	GetNum3DPolyhedra(ModelData *md);
extern unsigned int	GetNum3DPolyhedra_BufferPart1_Solids(ModelData *md);
extern unsigned int	GetNum3DPolyhedra_BufferPart2_RectangularSlices(ModelData *md);
extern unsigned int	GetNum3DPolyhedra_BufferPart3_CircularSlices(ModelData *md);
extern unsigned int	GetNum3DPolyhedra_BufferPart4_ClippedEllipticalSlices(ModelData *md);
extern void			Get3DPolyhedronPlacements(ModelData *md, unsigned int aPlacementBufferLength,
						TorusGames3DPolyhedronPlacementAsCArrays *aPlacementBuffer);
extern void			Get3DAxisAlignedBoundingBox(ModelData *md, double someBoundingBoxCornersInGameCell[2][4]);
extern void			InitBallMeshes(void);
extern void			InitTubeMeshes(void);
extern void			InitCircularSliceTriangleStrips(void);
extern void			RotateXYZ(double m[4][4]);
extern void			RotateZYX(double m[4][4]);
extern void			HalfTurnY(double m[4][4]);
extern void			SwapXY(double m[4][4]);
extern void			ReflectAxis(double m[4][4], unsigned int anAxis);

//	in TorusGamesOptions.c
extern void			ChangeHumanVsComputer(ModelData *md, bool aNewHumanVsComputer);
extern void			ChangeDifficultyLevel(ModelData *md, unsigned int aNewDifficultyLevel);
extern void			ChangeTopology(ModelData *md, TopologyType aNewTopology);
extern void			ChangeViewType(ModelData *md, ViewType aNewViewType);
extern void			SetNewGame(ModelData *md, GameType aNewGame);

//	in TorusGamesSimulation.c
extern void			ResetGame(ModelData *md);
extern void			ResetScrolling(ModelData *md);
extern void			SetUpGame(ModelData *md);
extern void			ShutDownGame(ModelData *md);
extern void			SimulationBegin(ModelData *md, SimulationType aSimulationStatus);
extern void			SimulationEnd(ModelData *md);
extern void			SnapToGrid3D(ModelData *md, double aMaxCoordDifference, double aDuration);
extern void			SnapToAxes3D(ModelData *md, double aMinCosine, double aDuration, bool aChangeToTilingFlag);
extern bool			Wall3DShouldBeTransparent(const double aWallNormalInFrameCell[4],
							/*const*/ double aFrameCellIntoWorld[4][4], const double aScaleFactor);
extern void			Reset3DGameWithAnimation(ModelData *md, ResetPurpose aResetPurpose, unsigned int aNewValue);

//	in TorusGamesMouse.c
extern void			MouseDown(ModelData *md, double aMouseH, double aMouseV, double aTimeStamp,
							bool aScrollFlag, bool aMarkFlag, bool aTemporalDoubleClick,
							bool aTwoFingerGestureFlag, bool aFlickGestureFlag);
extern void			MouseMove(ModelData *md, double aMouseDeltaH, double aMouseDeltaV, double aTimeStamp,
							bool aTwoFingerGestureFlag, bool aFlickGestureFlag);
extern void			MouseUp(ModelData *md, double aTimeStamp,
							bool aTwoFingerGestureFlag, bool aFlickGestureFlag, bool aTouchSequenceWasCancelled);
extern void			MouseGone(ModelData *md);
extern void			MouseWheel(ModelData *md, double aDeltaH, double aDeltaV);
extern void			MouseCancelled(ModelData *md, double aTimeStamp);
extern void			CoastingMomentum(ModelData *md, double aTimeInterval);
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
extern Placement2D	GetScaledHandCursorPlacement(ModelData *md);
#endif
extern TitledErrorMessage	*CharacterInput(ModelData *md, Char16 aCharacter);

//	in TorusGames2DMouse.c
extern void			MouseDown2D(ModelData *md, double aMouseH, double aMouseV, bool aScrollFlag, bool aMarkFlag, bool aTemporalDoubleClick,
						bool aTwoFingerGestureFlag, bool aFlickGestureFlag);
extern void			MouseMove2D(ModelData *md, double aMouseDeltaH, double aMouseDeltaV, double aMouseDeltaT,
						bool aTwoFingerGestureFlag, bool aFlickGestureFlag);
extern void			MouseUp2D(ModelData *md, double aDragDuration,
						bool aTwoFingerGestureFlag, bool aFlickGestureFlag, bool aTouchSequenceWasCancelled);
extern void			MouseGone2D(ModelData *md);
extern void			MouseWheel2D(ModelData *md, double aDeltaH, double aDeltaV);
extern void			CoastingMomentum2D(ModelData *md, double aTimeInterval);

//	in TorusGames3DMouse.c
extern void			MouseDown3D(ModelData *md, double aMouseH, double aMouseV, bool aScrollFlag,
						bool aTwoFingerGestureFlag, bool aFlickGestureFlag);
extern void			MouseMove3D(ModelData *md, double aMouseDeltaH, double aMouseDeltaV, double aMouseDeltaT,
						bool aTwoFingerGestureFlag, bool aFlickGestureFlag);
extern void			MouseUp3D(ModelData *md, double aDragDuration,
						bool aTwoFingerGestureFlag, bool aFlickGestureFlag, bool aTouchSequenceWasCancelled);
extern void			Normalize3DTilingIntoFrameCell(double aTilingIntoFrameCell[4][4], TopologyType aTopology);
extern void			Compute3DProjectionDerivative(double aLocationInWorld[4], double aProjectionDerivative[4][4]);
extern bool			Ray3DIntersectsSphere(HitTestRay3D *aRay, double aSphereCenter[3], double aSphereRadius, double *t);
extern bool			Ray3DIntersectsCube(HitTestRay3D *aRay, double aCubeCenter[3], double aCubeHalfWidth, double *t);
extern void			CoastingMomentum3D(ModelData *md, double aTimeInterval);

//	in TorusGames3DTiling.c
extern void			Make3DGameCellIntoTiling(double m[4][4], signed int x, signed int y, signed int z, TopologyType aTopology);

//	in TorusGamesUtilities.c
extern void			Normalize2DPlacement(Placement2D *aPlacement, TopologyType aTopology);
extern void			Normalize2DOffset(Offset2D *anOffset, TopologyType aTopology);
extern double		Shortest2DDistance(double h0, double v0, double h1, double v1, TopologyType aTopology, bool *aMirrorFlag);
extern Char16		BaseLetter(Char16 aLetter);
extern Char16		PromoteHalfWidthKatakana(Char16 aCharacter);
extern Char16		PromoteSmallKana(Char16 aCharacter);
extern Char16		RemoveTonos(Char16 aCharacter);
extern Char16		ConvertFinalSigma(Char16 aCharacter);
extern Char16		ComposeHiragana(Char16 aCharacter, Char16 *aPendingCharacter);
extern Char16		ComposeHangul(Char16 aCharacter, Char16 *aPendingCharacter, HangulSyllable *aPendingHangulSyllable);
extern Char16		QWERTYtoDubeolshik(Char16 aCharacter);
extern Char16		ComposeCyrillic(Char16 anExistingCharacter, Char16 aNewCharacter);
extern Char16		ComposeGreek(Char16 aCharacter);
extern Char16		ComposeLatin(Char16 aCharacter, Char16 *aPendingCharacter);
extern unsigned int	GetNumPuzzles(GameType aGame, TopologyType aTopology);
extern void			GetRawPuzzleData(GameType aGame, TopologyType aTopology, unsigned int aPuzzleIndex, Char16 *aBuffer, unsigned int aBufferLength);
extern void			RefreshStatusMessageText(ModelData *md);


//	in TorusGamesOptions.c
extern void			SetShowGlideAxes(ModelData *md, bool aShowGlideAxesChoice);
extern bool			GameIs3D(GameType aGame);
extern bool			TopologyIs3D(TopologyType aTopology);

//	in individual game files (TorusGames2DIntro.c, TorusGames2DTicTacToe.c, … )

extern void		     Intro2DSetUp(ModelData *md);
extern void		 TicTacToe2DSetUp(ModelData *md);
extern void		    Gomoku2DSetUp(ModelData *md);
extern void		      Maze2DSetUp(ModelData *md);
extern void		 Crossword2DSetUp(ModelData *md);
extern void		WordSearch2DSetUp(ModelData *md);
extern void		    Jigsaw2DSetUp(ModelData *md);
extern void		     Chess2DSetUp(ModelData *md);
extern void		      Pool2DSetUp(ModelData *md);
extern void		    Apples2DSetUp(ModelData *md);

extern void		 TicTacToe3DSetUp(ModelData *md);
extern void		      Maze3DSetUp(ModelData *md);


//	in TorusGames2DNone.c
extern unsigned int	GetNum2DNoneBackgroundTextureRepetitions(void);
extern void			Get2DNoneKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DNoneSprites(void);
extern void			Get2DNoneSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DIntro.c
extern unsigned int	GetNum2DIntroBackgroundTextureRepetitions(void);
extern void			Get2DIntroKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DIntroSprites(void);
extern void			Get2DIntroSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DTicTacToe.c
extern unsigned int	GetNum2DTicTacToeBackgroundTextureRepetitions(void);
extern void			Get2DTicTacToeKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DTicTacToeSprites(void);
extern void			Get2DTicTacToeSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DGomoku.c
extern unsigned int	GetNum2DGomokuBackgroundTextureRepetitions(void);
extern void			Get2DGomokuKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DGomokuSprites(void);
extern void			Get2DGomokuSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DMaze.c
extern unsigned int	GetNum2DMazeBackgroundTextureRepetitions(void);
extern void			Get2DMazeKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DMazeSprites(void);
extern void			Get2DMazeSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DCrossword.c
extern const Char16	*ChooseCrosswordCellFont(void);
extern unsigned int	GetNum2DCrosswordBackgroundTextureRepetitions(void);
extern void			Get2DCrosswordKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DCrosswordSprites(unsigned int aPuzzleSize);
extern void			Get2DCrosswordSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DWordSearch.c
extern const Char16	*ChooseWordSearchCellFont(void);
extern unsigned int	GetNum2DWordSearchBackgroundTextureRepetitions(void);
extern void			Get2DWordSearchKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DWordSearchSprites(unsigned int aPuzzleSize, unsigned int aNumLinesFound);
extern void			Get2DWordSearchSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DJigsaw.c
extern unsigned int	GetNum2DJigsawBackgroundTextureRepetitions(void);
extern void			Get2DJigsawKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DJigsawSprites(unsigned int aPuzzleSize);
extern void			Get2DJigsawSpritePlacementsOrTexturePlacements(ModelData *md, unsigned int aNumSprites,
								Placement2D *aPlacementBuffer, double (*aTexturePlacementBuffer)[4]);

//	in TorusGames2DChess.c
extern unsigned int	GetNum2DChessBackgroundTextureRepetitions(void);
extern void			Get2DChessKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DChessSprites(void);
extern void			Get2DChessSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DPool.c
extern unsigned int	GetNum2DPoolBackgroundTextureRepetitions(void);
extern void			Get2DPoolKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DPoolSprites(void);
extern void			Get2DPoolSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames2DApples.c
extern const Char16	*ChooseApplesNumeralFont(void);
extern unsigned int	GetNum2DApplesBackgroundTextureRepetitions(void);
extern void			Get2DApplesKleinAxisColors(float someKleinAxisColors[2][4]);
extern unsigned int	GetNum2DApplesSprites(unsigned int aBoardSize);
extern void			Get2DApplesSpritePlacements(ModelData *md, unsigned int aNumSprites, Placement2D *aPlacementBuffer);

//	in TorusGames3DTicTacToe.c
extern unsigned int	GetNum3DTicTacToePolyhedra_BufferPart1_Solids(ModelData *md);
extern unsigned int	GetNum3DTicTacToePolyhedra_BufferPart2_RectangularSlices(ModelData *md);
extern unsigned int	GetNum3DTicTacToePolyhedra_BufferPart3_CircularSlices(ModelData *md);
extern unsigned int	GetNum3DTicTacToePolyhedra_BufferPart4_ClippedEllipticalSlices(ModelData *md);
extern void			Get3DTicTacToePolyhedronPlacementsForSolids(ModelData *md, unsigned int aPlacementBufferLength,
						TorusGames3DPolyhedronPlacementAsCArrays *aPlacementBuffer);
extern void			Get3DTicTacToePolyhedronPlacementsForOneSlice(
						ModelData *md, unsigned int anAxis, double anIntercept, double aFrameCellCenterInGameCell[4],
						double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
						unsigned int *aRectangularSlicePlacementBufferLengthPtr,		TorusGames3DPolyhedronPlacementAsCArrays **aRectangularSlicePlacementBufferPtr,
						unsigned int *aCircularSlicePlacementBufferLengthPtr,			TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr,
						unsigned int *aClippedEllipticalSlicePlacementBufferLengthPtr,	TorusGames3DPolyhedronPlacementAsCArrays **aClippedEllipticalSlicePlacementBufferPtr);
extern void			Get3DTicTacToeAxisAlignedBoundingBox(ModelData *md, double someBoundingBoxCornersInGameCell[2][4]);
extern bool			WinLineIsCircular(TopologyType aTopology, WinLine *aWinLine);

//	in TorusGames3DMaze.c
extern unsigned int	GetNum3DMazePolyhedra_BufferPart1_Solids(ModelData *md);
extern unsigned int	GetNum3DMazePolyhedra_BufferPart2_RectangularSlices(ModelData *md);
extern unsigned int	GetNum3DMazePolyhedra_BufferPart3_CircularSlices(ModelData *md);
extern unsigned int	GetNum3DMazePolyhedra_BufferPart4_ClippedEllipticalSlices(ModelData *md);
extern void			Get3DMazePolyhedronPlacementsForSolids(ModelData *md, unsigned int aPlacementBufferLength,
						TorusGames3DPolyhedronPlacementAsCArrays *aPlacementBuffer);
extern void			Get3DMazePolyhedronPlacementsForOneSlice(
						ModelData *md, unsigned int anAxis, double anIntercept, double aFrameCellCenterInGameCell[4],
						double aRotationalPartOfSlicePlacement[4][4], double aGameCellIntoFrameCell[4][4],
						unsigned int *aRectangularSlicePlacementBufferLengthPtr,		TorusGames3DPolyhedronPlacementAsCArrays **aRectangularSlicePlacementBufferPtr,
						unsigned int *aCircularSlicePlacementBufferLengthPtr,			TorusGames3DPolyhedronPlacementAsCArrays **aCircularSlicePlacementBufferPtr);
extern void			Get3DMazeAxisAlignedBoundingBox(ModelData *md, double someBoundingBoxCornersInGameCell[2][4]);

